﻿using System;
using System.IO.Ports;
using System.Threading;

namespace MicroRWD.Common
{
    // Encapsulation of the basic serial port
    public class Port : IDisposable
    {
        #region Private Constants

        // Port setup - 9600 baud
        private const int C_BAUDRATE = 9600;

        // Port setup - No parity
        private const Parity C_PARITY = Parity.None;
        
        // Port setup - 8 data bits
        private const int C_DATABITS = 8;
        
        // Port setup - 1 stop bit
        private const StopBits C_STOPBITS = StopBits.One;

        // Maximum reply size
        private const int C_MAX_REPLY_LEN = 512;

        // Intra message timeout ms
        private const int C_INTRA_MESSAGE_TIMEOUT_MS = 50;

        #endregion


        #region Private Properties

        // The event used to indicate a CTS assertion
        private ManualResetEvent CTS_Event { get; set; }

        // The raw serial port instance associated with the Port
        private SerialPort SerialPort { get; set; }

        #endregion


        #region Public Properties

        // Return the port name associated with the serial port
        public string PortName { get { return SerialPort.PortName; } }

        // Return true if the serial port is open
        public bool IsOpen { get { return SerialPort.IsOpen; } }

        // Return true if the CTS signal is asserted
        public bool IsCTS { get { return SerialPort.CtsHolding; } }

        #endregion


        #region Constructor

        // Constructs a new Port using the specified port name
        public Port(string _portName)
        {
            // Create the SerialPort using the specified port name and constant port settings
            SerialPort = new SerialPort(_portName, C_BAUDRATE, C_PARITY, C_DATABITS, C_STOPBITS);

            // Initialise event used to signal CTS assertion
            CTS_Event = new ManualResetEvent(false);

            try
            {
                // Open the underlying serial port
                SerialPort.Open();

                // Assert DTR
                SerialPort.DtrEnable = true;

                // Monitor pin change events so that we can detect CTS assertion accurately
                SerialPort.PinChanged += new SerialPinChangedEventHandler(MySerialPinChangedEventHandler);

                // Disable hardware handshaking (we monitor the CTS line directly)
                SerialPort.Handshake = Handshake.None;
            }
            catch (Exception ex)
            {
                Log.Warning(ex.Message);
            }
        }

        #endregion


        #region Public Methods

        // Wait for the MicroRWD to pull CTS low (or timeout)
        public bool WaitForCTS(int _ctsTimeoutMillis)
        {
            // Placeholder for the result
            bool result = false;

            // Don't poll an unconnected port
            if (SerialPort.IsOpen)
            {
                // Wait until the CTS signal is asserted or the timeout expires
                result = CTS_Event.WaitOne(_ctsTimeoutMillis);
            }
            else
            {
                // Error
                Log.Error("port is not open");
            }

            // Return true if signalled (CTS asserted)
            return result;
        }

        // Send bytes to the RWD
        public void SendCmd(byte[] _data)
        {
            // Don't send to an unconnected port
            if (SerialPort.IsOpen)
            {
                // Don't send null or empty data
                if ((_data != null) && _data.Length > 0)
                {
                    // Empty input buffer
                    SerialPort.DiscardInBuffer();

                    // Transmit command via the serial port
                    SerialPort.Write(_data, 0, _data.Length);

                    // Information
                    Log.Information(String.Join(string.Empty, Array.ConvertAll(_data, b => b.ToString("X2"))));
                }
                else
                {
                    // Error
                    Log.Error("null or empty data");
                }
            }
            else
            {
                // Error
                Log.Error("port is not open");
            }
        }

        // Read byte(s) from the RWD (or timeout in milliseconds)
        public byte[] GetReply(int _commandTimeoutMillis)
        {
            // Zero length reply on timeout, data otherwise
            byte[] result = new byte[0];
            int nread = 0;

            // Don't read from an unconnected port
            if (SerialPort.IsOpen)
            {
                // Somewhere to hold the data read
                byte[] buffer = new byte[C_MAX_REPLY_LEN];

                // Set read timeout on serial port
                SerialPort.ReadTimeout = _commandTimeoutMillis;

                // Read until we receive the required number of bytes (or timeout)
                while (nread < C_MAX_REPLY_LEN)
                {
                    try
                    {
                        // Read a single byte (or timeout)
                        int rc = SerialPort.Read(buffer, nread, 1);

                        // Increment nread
                        nread += rc;

                        // Shorten timeout after first character has been received
                        if (nread == 1)
                        {
                            SerialPort.ReadTimeout = Math.Min(C_INTRA_MESSAGE_TIMEOUT_MS, _commandTimeoutMillis);
                        }
                    }
                    catch (TimeoutException)
                    {
                        // Timeout - terminate while loop
                        break;
                    }
                }

                // Copy read data to result (if anything was read)
                if ((nread > 0) && (nread < C_MAX_REPLY_LEN))
                {
                    // Copy received data into result array
                    result = new byte[nread];
                    Array.Copy(buffer, result, nread);

                    // Information
                    Log.Information(String.Join(string.Empty, Array.ConvertAll(result, b => b.ToString("X2"))));
                }
            }
            else
            {
                // Error
                Log.Error("port is not open");
            }

            // Return result
            return result;
        }

        #region IDisposable Methods

        // Dispose of the underlying resources
        public void Dispose()
        {
            // If the serial port is still open, close it
            if (SerialPort.IsOpen)
            {
                SerialPort.Close();
            }

            // Dispose of the resources allocated by the serial port
            SerialPort.Dispose();

            // Prevent further finalisation
            GC.SuppressFinalize(this);
        }
        
        #endregion

        #endregion


        #region Private Methods

        // Event handler for the pin changed event - used to monitor CTS assertion
        private void MySerialPinChangedEventHandler(object sender, SerialPinChangedEventArgs e)
        {
            switch (e.EventType)
            {
                case SerialPinChange.CtsChanged:
                    if (IsCTS)
                    {
                        CTS_Event.Set();
                    }
                    else
                    {
                        CTS_Event.Reset();
                    }
                    break;

                default:
                    break;
            }
        }

        #endregion

    }
}
